#!/usr/bin/env bash

ARCH=$ARCH
CROSS_COMPILE=$CROSS_COMPILE
TARGETS=
HOST_ARCH=`uname -m`
CMIT_NEW=HEAD
CMIT_SAVED=`git branch | grep ^* | awk '{print $2}'`
WHITEL_F=
CONFIG_F=

GIT_STASHED=false

trap 'do_ctrlc' INT

do_exit()
{
	git checkout $CMIT_SAVED &>/dev/null
	if [ $GIT_STASHED == true ]; then
		git stash pop
	fi
	exit $1
}

do_ctrlc()
{
	echo_colo "Ctrl+C is captured!" red
	do_exit 0
}

echo_colo()
{
	case $2 in
		red)	echo -e "\033[31m$1\033[0m";;
		green)	echo -e "\033[32m$1\033[0m";;
		yellow)	echo -e "\033[33m$1\033[0m";;
		blue)	echo -e "\033[34m$1\033[0m";;
		*)		echo -e "\033[30m$1\033[0m";;
	esac
}

usage()
{
	echo "Usage: $0 [-a ARCH] [-w WHITELIST] [-n NEW] [-c CONFIG]"
	echo "  -a, --arch          Specify ARCH"
	echo "  -c, --config        Specify kernel config to build kernel"
	echo "  -h, --help          Help"
	echo "  -n, --new           New commit"
	echo "  -w, --whitelist     Specify whitelist file"
	echo " "
	echo "For detail instructions, please see : "
	echo "<http://pm.kylin.com/biz/task-view-17942.html>"
}

if [ x"" = x"`which dump_abi`" ]; then
	echo_colo "Please install abigail tools first!" red && do_exit 1
fi

ARGS=`getopt -o a:c:h::n:w: --long arch:,config:,help::,new:,whitelist: -n '$0' -- "$@"`

if [ $? != 0 ]; then
    echo "Terminating..."
    do_exit 1
fi

eval set -- "${ARGS}"

while true; do
	case "$1" in
		-h|--help)
			usage
			do_exit 0
			;;
		-a|--arch)
			ARCH=$2
			shift 2
			;;
		-n|--new)
			CMIT_NEW=`git show --name-only --oneline $2 2>/dev/null | awk 'NR==1 { print $1}'`
			shift 2
			;;
		-c|--config)
			CONFIG_F=$2
			shift 2
			;;
		-w|--whitelist)
			WHITEL_F=$2
			shift 2
			;;
		--)
			shift
			break
			;;
		*)
			echo "Internal error!"
			do_exit 1
			;;
	esac
done

if [ x"$ARCH" = x"" ]; then
	if [ x"$CONFIG_F" != x"" ]; then
		ARCH=`echo "$CONFIG_F" | awk '{ print gensub(/arch\/(.*)\/configs\/(.*)/, "\\\\1", "g") }'`
		if [ x"$ARCH" != x"arm64" -a x"$ARCH" != x"mips" -a x"$ARCH" != x"x86" ]; then
			ARCH=$HOST_ARCH
		else
			echo_colo "=== Get ARCH=${ARCH} from config_file path <$CONFIG_F>" yellow
		fi
	else
		ARCH=$HOST_ARCH
	fi
fi

case $ARCH in
	arm64|aarch64)
		ARCH=arm64
		Arch=aarch64
		if [ x"$HOST_ARCH" != x"aarch64" -a x"$CROSS_COMPILE" = x"" ]; then
			CROSS_COMPILE=aarch64-linux-gnu-
		fi
		TARGETS="Image modules"
		;;
	mips|mips64el)
		ARCH=mips
		Arch=mips64el
		if [ x"$HOST_ARCH" != x"mips64el" -a x"$CROSS_COMPILE" = x"" ]; then
			CROSS_COMPILE=mips-linux-gnu-
		fi
		TARGETS="vmlinuz modules"
		;;
	x86|x86_64)
		ARCH=x86
		Arch=x86_64
		if [ x"$HOST_ARCH" != x"x86_64" -a x"$CROSS_COMPILE" = x"" ]; then
			CROSS_COMPILE=x86_64-linux-gnu-
		fi
		TARGETS="vmlinux modules"
		;;
	*)
		echo_colo "=== Unknow ARCH=${ARCH}" red && do_exit 1
		;;
esac

# set default baseline commit id here
CMIT_BASE=`git log  -n 1  --pretty="%H" kabi/Module.kabi_${Arch}`

if [ x"$CONFIG_F" = x"" ]; then
	CONFIG_F="arch/${ARCH}/configs/kylinos_generic_defconfig"
fi

if [ ! -f $CONFIG_F ]; then
	echo_colo "=== config file <$CONFIG_F> invalid!" red && do_exit 1
fi

if [ -z "$WHITEL_F" ]; then
	if [ -f kabi/whitelist_${ARCH} ]; then
		WHITEL_F=kabi/whitelist_${ARCH}
	else
		echo_colo "=== kabi/whitelist_${ARCH} not found, use parameter --whitelist to specify whitelist!" red \
			&& do_exit 1
	fi
else
	if [ ! -f $WHITEL_F ]; then
		echo_colo "=== whitelist file <$WHITEL_F> invalid!" red && do_exit 1
	fi
fi

export ARCH=${ARCH}
export CROSS_COMPILE=${CROSS_COMPILE}

LOGFILE=

run_make()
{
        KBUILD_PRESERVE=1 KBUILD_SYMTYPES=1  make $@ &>$LOGFILE
}

do_compile()
{
	echo -n "=== Checkout commit $1, "
	if [ x"$1" != x"HEAD" ]; then
		git checkout . &>/dev/null && git checkout $1 &>/dev/null
		if [ $? != 0 ]; then
			echo_colo "failed!" red
			echo_colo "Please try <git checkout . && git checkout $1> manually" blue
			return 1
		fi
	fi
	echo_colo "done!" green

	BUILDDIR="build-${ARCH}-$1"
	echo -n "=== Create output dir $BUILDDIR, "
	mkdir $BUILDDIR &>/dev/null
	if [ -d $BUILDDIR ]; then
		echo_colo "done!" green
	else
		echo_colo "failed!" red
		return 1
	fi

	echo -n "=== Check config file <$CONFIG_F>, "
	if [ ! -f $CONFIG_F ]; then
		echo_colo "file not exist!" red
		return 1
	fi
	cp $CONFIG_F ${BUILDDIR}/.config
	sed -i 's/# CONFIG_MODVERSIONS is not set/CONFIG_MODVERSIONS=y/' ${BUILDDIR}/.config
	sed -i 's/# CONFIG_DEBUG_INFO is not set/CONFIG_DEBUG_INFO=y/' ${BUILDDIR}/.config
	echo_colo "done!" green

	LOGFILE="/tmp/build-${ARCH}-`whoami`-$1.log"
	echo -n "=== Build commit [$1], Log to <$LOGFILE>, "
	NPROC=$((`nproc`*2))
	run_make O=${BUILDDIR}/ olddefconfig && \
		run_make O=${BUILDDIR}/ -j${NPROC} ${TARGETS}
	ret=$?

	if [ x"$1" == x"HEAD" ]; then
		git stash
		GIT_STASHED=true
	fi

	if [ $ret != 0 ]; then
		echo_colo "failed!" red
		return 1
	else
		echo_colo "done!" green
		rm $LOGFILE
		echo "=== Dump ABI to file <${BUILDDIR}/abi.xml>"
		dump_abi --kmi-whitelist ${WHITEL_F} \
			--linux-tree ${BUILDDIR}
		return 0
	fi
}

BASABI_F="build-${ARCH}-${CMIT_BASE}/abi.xml"
NEWABI_F="build-${ARCH}-${CMIT_NEW}/abi.xml"

echo "=== Prepare to build new commit [$CMIT_NEW]"
do_compile $CMIT_NEW || do_exit 1

echo "=== Prepare to build baseline commit [$CMIT_BASE]"
if [ ! -f $BASABI_F ]; then
	do_compile $CMIT_BASE || do_exit 1
fi

ret=0
if [ -f $BASABI_F -a -f $NEWABI_F ]; then
	diff_abi --baseline $BASABI_F --new $NEWABI_F \
       		--kmi-symbol-list ${WHITEL_F} --report report.out
	ret=$?
	if [ $ret -eq 0 ]; then
		echo_colo "=== KABI SUCESS !!!!" green
	else
		echo_colo "=== KABI Failed, Please check ABI diff file <report.out>" red
	fi
fi

# If abi changed, return 8
do_exit $ret
