/*
 * EC (Embedded Controller) WPCE775L device driver header
 *
 * Copyright (C) 2011 Lemote Inc.
 * Author : Wang Rui <wangr@lemote.com>
 * Author : Huang Wei <huangw@lemote.com>
 * Date   : 2011-02-21
 * Author : xiongxin <xiongxin@tj.kylinos.cn>
 * Date	  : 2020-05-27
 * EC relative header file. All the EC registers should be defined here.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at you option) and later version.
 */

#ifndef __EC_WPCE775L_H__
#define __EC_WPCE775L_H__

#define EC_VERSION		"1.11"

/* EC delay times to wait */
#define EC_SEND_TIMEOUT		0xffff
#define EC_RECV_TIMEOUT		0xffff

/* EC access port for with Host communication. */
#define EC_CMD_PORT			0x66
#define EC_STS_PORT			0x66
#define EC_DAT_PORT			0x62

/* ACPI legacy commands. */
#define CMD_READ_EC			0x80	/* Read EC command. */
#define CMD_WRITE_EC		0x81	/* Write EC command. */
/* Query EC command, for get SCI event number. */
#define CMD_GET_EVENT_NUM	0x84

/* FAn speed level. [0,5] or [0x06, 0x38]*/
#define INDEX_FAN_SPEED_LEVEL	0x5C
/* Fan control mode, 0 = by EC, 1 = by Host.*/
#define INDEX_FAN_CTRLMOD		0x5D
enum {
	FAN_CTRL_BYEC = 0,
	FAN_CTRL_BYHOST
};

#define INDEX_FAN_SPEED_LOW		0x08	/* Fan speed low byte.*/
#define INDEX_FAN_SPEED_HIGH	0x09	/* Fan speed high byte. */

/* LCD backlight control mode: 0 = by EC, 1 = by HOST */
#define INDEX_BACKLIGHT_CTRLMODE	0x57
enum {
	BACKLIGHT_CTRL_BYEC = 0,
	BACKLIGHT_CTRL_BYHOST
};

/* LCD backlight status or control: 0 = turn off, 1 = turn on */
#define INDEX_BACKLIGHT_STSCTRL		0x58
enum
{
	BACKLIGHT_OFF = 0,
	BACKLIGHT_ON
};
/* LCD backlight brightness max level */
#define	INDEX_DISPLAY_MAXBRIGHTNESS_LEVEL	0x59
/* 10 stages (0~9) LCD backlight brightness adjust */
#define	INDEX_DISPLAY_BRIGHTNESS	0x5A

#define INDEX_TOUCHPAD_ENABLE_LED	0x56
enum {
	TP_EN_LED_OFF,
	TP_EN_LED_ON
};

/*
 * The reported battery die temperature.
 * The temperature is expressed in units of 0.25 seconds and is updated
 * every 2.56 seconds.
 * The equation to calculate reported pack temperature is:
 * Temperature = 0.1 * (256 * TEMPH + TEMPL) Kelvin
 * Temperature -= 273 Degrees Celsius
 * The host sytem has read-only access to this register pair.
 */
/* Battery temperature byte. */
#define INDEX_BATTERY_TEMP_LOW		0x20
#define INDEX_BATTERY_TEMP_HIGH		0x21
/* Battery Voltage byte. */
#define INDEX_BATTERY_VOL_LOW		0x22
#define INDEX_BATTERY_VOL_HIGH		0x23
/* Battery Current byte. */
#define INDEX_BATTERY_CURRENT_LOW	0x24
#define INDEX_BATTERY_CURRENT_HIGH	0x25
/* Battery AverageCurrent byte. */
#define INDEX_BATTERY_AC_LOW		0x26
#define INDEX_BATTERY_AC_HIGH		0x27
/* Battery RemainingCapacity percent. */
#define INDEX_BATTERY_CAPACITY		0x2A
/* Battery RemainingCapacity byte. */
#define INDEX_BATTERY_RC_LOW		0x2E
#define INDEX_BATTERY_RC_HIGH		0x2F
/* Battery AverageTimeToEmpty byte. */
#define INDEX_BATTERY_ATTE_LOW		0x30
#define INDEX_BATTERY_ATTE_HIGH		0x31
/* Battery AverageTimeToFull byte. */
#define INDEX_BATTERY_ATTF_LOW		0x32
#define INDEX_BATTERY_ATTF_HIGH		0x33
/* Battery FullChargeCapacity byte. */
#define INDEX_BATTERY_FCC_LOW		0x34
#define INDEX_BATTERY_FCC_HIGH		0x35
/* Battery ChargingVoltage byte. */
#define INDEX_BATTERY_CV_LOW		0x38
#define INDEX_BATTERY_CV_HIGH		0x39
/* Battery CycleCount byte. */
#define INDEX_BATTERY_CYCLECNT_LOW	0x3C
#define INDEX_BATTERY_CYCLECNT_HIGH	0x3D

/* Battery static information. */
/* Battery DesignCapacity byte. */
#define INDEX_BATTERY_DC_LOW		0x60
#define INDEX_BATTERY_DC_HIGH		0x61
/* Battery DesignVoltage byte. */
#define INDEX_BATTERY_DV_LOW		0x62
#define INDEX_BATTERY_DV_HIGH		0x63
/* Battery ManufactureDate byte. */
#define INDEX_BATTERY_MFD_LOW		0x64
#define INDEX_BATTERY_MFD_HIGH		0x65
/* Battery SerialNumber byte. */
#define INDEX_BATTERY_SN_LOW		0x66
#define INDEX_BATTERY_SN_HIGH		0x67
/* Battery ManufacturerName string length. */
#define INDEX_BATTERY_MFN_LENG		0x68

/* Battery DeviceName string length. */
#define INDEX_BATTERY_DEVNAME_LENG	0x74
/* Battery DeviceChemitry string length. */
#define INDEX_BATTERY_DEVCHEM_LENG	0x7C

#define MASK(x)	(1 << x)

#define INDEX_POWER_STATUS		0xA2	/* Read current power status. */
enum {
	/* Power-on start status, 1 = on power-on, 0 = power-on complete. */
	BIT_POWER_PWRON = 0,
	BIT_POWER_BATVL,		/* Battery in very low status. */
	BIT_POWER_BATL,			/* Battery in low status. */
	BIT_POWER_BATFCHG,		/* Battery in fully charging status. */
	BIT_POWER_BATCHG,		/* Battery in charging status. */
	BIT_POWER_TERMINATE,	/* Battery in terminate charging status. */
	BIT_POWER_BATPRES,		/* Battery present. */
	BIT_POWER_ACPRES		/* AC present. */
};

#define INDEX_DEVICE_STATUS		0xA3	/* Read Current Device Status */
enum {
	BIT_DEVICE_TP = 0,	/* TouchPad status: 0 = close, 1 = open */
	BIT_DEVICE_WLAN,	/* WLAN status: 0 = close, 1 = open */
	BIT_DEVICE_3G,		/* 3G status: 0 = close, 1 = open */
	BIT_DEVICE_CAM,		/* Camera status: 0 = close, 1 = open */
	BIT_DEVICE_MUTE,	/* Mute status: 0 = close, 1 = open */
	BIT_DEVICE_LID,		/* LID status: 0 = close, 1 = open */
	BIT_DEVICE_BKLIGHT,	/* BackLight status: 0 = close, 1 = open */
	BIT_DEVICE_SIM		/* SIM Card status: 0 = pull out, 1 = insert */
};

/* Camera control */
#define INDEX_CAM_STSCTRL		0xAA
enum {
	CAM_STSCTRL_OFF = 0,
	CAM_STSCTRL_ON
};

/* data destroy led control */
#define INDEX_DATA_DESTROY		0xB0
enum {
	DATA_DESTROY_OFF = 0,
	DATA_DESTROY_ON
};

/* The led of board healthy */
#define INDEX_BOARD_HEALTHY		0xB1

#define EC_IBF	(1 << 1) /* 1 = Input buffer full (data ready for ec) */
#define EC_OBF	(1 << 0) /* 1 = Output buffer full (data ready for host) */

/* SCI Event Number from EC */
enum {
	SCI_EVENT_NUM_WLAN = 0x21,	/* 0x21, Fn+F1, Wlan is on or off */
	SCI_EVENT_NUM_3G,	/* 0x22, Fn+F9 for 3G switch */
	SCI_EVENT_NUM_LID,	/* 0x23, press the lid or not */
	SCI_EVENT_NUM_DISPLAY_TOGGLE,	/* 0x24, Fn+F8 for display switch */
	/* 0x25, Fn+ESC for entering sleep mode */
	SCI_EVENT_NUM_SLEEP,
	/* 0x26, Fn+F3, LCD backlight brightness up adjust */
	SCI_EVENT_NUM_BRIGHTNESS_UP,
	/* 0x27, Fn+F2, LCD backlight brightness down adjust */
	SCI_EVENT_NUM_BRIGHTNESS_DN,
	SCI_EVENT_NUM_CAMERA,	/* 0x28, Fn+F10, Camera is on or off */
	SCI_EVENT_NUM_TP,		/* 0x29, Fn+F11, TouchPad is on */
	SCI_EVENT_NUM_AUDIO_MUTE,	/* 0x2A, Fn+F4, Mute is on or off */
	/* 0x2B, Fn+F7, Black screen is on or off */
	SCI_EVENT_NUM_BLACK_SCREEN,
	SCI_EVENT_NUM_VOLUME_UP,	/* 0x2C, Fn+F6, Volume up adjust */
	SCI_EVENT_NUM_VOLUME_DN,	/* 0x2D, Fn+F5, Volume down adjust */
	SCI_EVENT_NUM_OVERTEMP,		/* 0x2E, Over-temperature happened */
	SCI_EVENT_NUM_SIM,			/* 0x2F, SIM Card Detect */
	SCI_EVENT_NUM_AC,			/* 0x30, AC in/out */
	SCI_EVENT_NUM_BAT,			/* 0x31, BAT in/out */
	/* 0x32, Battery Low capacity alarm, < 10% */
	SCI_EVENT_NUM_BATL,
	/* 0x33, Battery VeryLow capacity alarm, < 5% */
	SCI_EVENT_NUM_BATVL,
	/* 0x34, CPU Throttling event alarm, CPU Temperature > 85 or < 80. */
	SCI_EVENT_NUM_THROT,
	SCI_EVENT_NUM_POWER = 0x37,	/* 0x37, Power button */
	SCI_EVENT_RESOLUTION_SETTING,	/* 0x38, Resolution Setting */
	SCI_EVENT_MEDIA_RUN_PAUSE,	/* 0x39, Media Play or Pause */
	SCI_EVENT_MEDIA_STOP,		/* 0x3A, Media Stop */
	SCI_EVENT_MEDIA_LAST,		/* 0x3B, Media Play last one */
	SCI_EVENT_MEDIA_NEXT,		/* 0x3C, Media Play next one */
	SCI_EVENT_RECOVERY = 0x3D	/* 0x3D, Recovery Event */
};

#define SCI_EVENT_NUM_START		SCI_EVENT_NUM_WLAN
#define SCI_EVENT_NUM_END		SCI_EVENT_RECOVERY

typedef int (*sci_handler)(int status);

/* The general ec index-io port read action */
extern unsigned char ec_read(unsigned char index);
extern unsigned char ec_read_all(unsigned char command,
		unsigned char index);
extern unsigned char ec_read_noindex(unsigned char command);

/* The general ec index-io port write action */
extern int ec_write(unsigned char index, unsigned char data);
extern int ec_write_all(unsigned char command, unsigned char index,
		unsigned char data);
extern int ec_write_noindex(unsigned char command, unsigned char data);

/* Query sequence of 62/66 port access routine. */
extern int ec_query_seq(unsigned char command);
extern int ec_get_event_num(void);

extern void clean_ec_event_status(void);
#endif /* __EC_WPCE775L_H__ */
