/** @file semem_com.h
 *  @brief This file includes some basic definitions
 *  @author Montage
 *  @date   4/20/2018
 */
#ifndef __SEMEM_COM_H__
#define __SEMEM_COM_H__		1

#ifndef __KERNEL__
#include <sys/types.h>
#include <unistd.h>

#ifndef roundup
#define roundup(x, y)	((((x) + (y) - 1) / (y)) * (y))
#endif

#ifndef rounddown
#define rounddown(x, y)	(((x) / (y)) * (y))
#endif

#ifndef PAGE_SIZE
#define PAGE_SIZE		(1 << 12)
#endif

#else
#include <linux/types.h>
#endif


/**
 * @brief definitions of security attributes
 */
enum semem_attribute{
	MEM_ATTR_NONE,			/**< the attribute allows both read and write operation  */

	MEM_PROT_READ,			/**< the attribute allows only read operation  */

	MEM_PROT_WRITE,			/**< the attribute allows only write operation */

	MEM_PROT_HIDDEN,		/**< the attribute does not allow neither read nor write operation  */

	MEM_TRACE_READ,			/**< the attribute only traces read operation and allows read and write operation */
	MEM_TRACE_WRITE,		/**< the attribute only traces write operation and allows read and write operation  */
	MEM_TRACE_RDWR,			/**< the attribute only traces both read and operation and allows read and write operation  */

	MEM_ATTR_MAX
};

enum semem_status{
	SEMEM_STATUS_MIN,
	SEMEM_ENABLE,
	SEMEM_DISABLE,
	SEMEM_STATUS_MAX
};

enum semem_log_op{
	SEMEM_LOG_PROC_ADDR,
	SEMEM_LOG_DRAM_ADDR
};

#ifndef _RANGE_
#define _RANGE_
/**
 * @brief Basic range definition
 */
struct range_struct{
	unsigned long start;		/**< start address of the range  */
	unsigned long size;			/**< size of the range  */
};

static inline unsigned long range_end(struct range_struct *range){
	return range->start + range->size - 1;
}

/**
 * @brief Basic security range definition
 */
struct semem_range{
	pid_t	pid;				/**< the owner of security range  */
	unsigned long start;		/**< start address of the security range  */
	unsigned long size;			/**< size of the security range  */
	unsigned long flags;		/**< security attribute of the range, @sa semem_attribute  */
};
#endif

#define SEMEM_ATTR_SHIFT	0x0
#define SEMEM_STATUS_SHIFT	0x4

#define SEMEM_ATTR_MASK		0x0F
#define SEMEM_STATUS_MASK	0x0F

#define RULE_GET_READ_LOG	0
#define RULE_GET_WRITE_LOG	1

static inline unsigned long semem_range_flags(unsigned long attr, unsigned long status){
	return (attr << SEMEM_ATTR_SHIFT) | (status << SEMEM_STATUS_SHIFT);
}


static inline unsigned long semem_range_attr(struct semem_range *range){
	return (range->flags & SEMEM_ATTR_MASK);
}

static inline unsigned long semem_range_status(struct semem_range *range){
	return (range->flags >> SEMEM_STATUS_SHIFT) & SEMEM_STATUS_MASK;
}

/**
 * @brief  get the end address of the security range specified by a
 * @param  a [input] the pointer that point to the range
 * @return the end address of the security range
 */
static inline unsigned long semem_range_end(struct semem_range *a){
	return a->start + a->size - 1;
}

/**
 * @brief  get the end address of memory range specified by a
 * @param  a [input] the pointer that points to the range
 * @return the end address of the security range
 */
static inline unsigned long com_range_end(struct range_struct *a){
	return a->start + a->size - 1;
}

/**
 * @brief  compare the ranges for overlap
 * @param  a [input] the pointer that points to the range
 * @param  b [input] the pointer that points to another range
 * @return 1, two ranges are overlap
 * 0, two ranges are not overlap
 */
static inline int is_range_overlap(struct range_struct *a, struct range_struct *b){
	return ((a->start >= b->start) && (a->start < (b->start + b->size))) ||
			((b->start >= a->start) && (b->start < (a->start + a->size)));
}

/**
 * @brief  compare the ranges for equality
 * @param  a [input] the pointer that points to the range
 * @param  b [input] the pointer that points to another range
 * @return 1, two ranges are equal
 * 0, two ranges are not equal
 */
static inline int is_range_equal(struct range_struct *a, struct range_struct *b){
	return ((a->start == b->start) && (a->size == b->size));
}

/**
 * @brief  compare the security ranges for equality
 * @param  a [input] the pointer that points to the range
 * @param  b [input] the pointer that points to another range
 * @return 1, two ranges are equal
 * 0, two ranges are not equal
 */
static inline int is_semem_range_equal(struct semem_range *a, struct semem_range *b){
	return ((a->start == b->start) && (a->size == b->size));
}

/**
 * @brief  check if the range is valid in a straightforward way
 * @param  a [input] the pointer that points to the range
 * @return 1, range is valid
 * 0, range is not valid
 */
static inline int range_valid_check(struct range_struct *a){
	return (a->start != 0 && a->size != 0);
}

/**
 * @brief  check if the security range is valid in a straightforward way
 * @param  a [input] the pointer that points to the range
 * @return 1, range is valid
 * 0, range is not valid
 */
static inline int semem_range_valid_check(struct semem_range *a){
	return (a->start != 0 && a->size != 0);
}

/**
 * @brief  get the range information from a security range
 * @param  se_range [input] the pointer that points to the security range
 * @param  range [output] the pointer that points to the range
 */
static inline void get_semem_range(struct semem_range *se_range, struct range_struct *range){
	range->start = se_range->start;
	range->size = se_range->size;
}

#endif

