/*
 * Copyright 2010 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 */

#ifndef _ASM_TILE_DMA_MAPPING_H
#define _ASM_TILE_DMA_MAPPING_H

#include <linux/mm.h>
#include <linux/scatterlist.h>
#include <linux/cache.h>
#include <linux/io.h>

#ifdef __tilegx__
#define ARCH_HAS_DMA_GET_REQUIRED_MASK
#endif

extern const struct dma_map_ops *tile_dma_map_ops;
extern const struct dma_map_ops *gx_pci_dma_map_ops;
extern const struct dma_map_ops *gx_legacy_pci_dma_map_ops;
extern const struct dma_map_ops *gx_hybrid_pci_dma_map_ops;

static inline const struct dma_map_ops *get_dma_ops(struct device *dev)
{
	if (dev && dev->archdata.dma_ops)
		return dev->archdata.dma_ops;
	else
		return tile_dma_map_ops;
}

static inline dma_addr_t get_dma_offset(struct device *dev)
{
	return dev->archdata.dma_offset;
}

static inline void set_dma_offset(struct device *dev, dma_addr_t off)
{
	dev->archdata.dma_offset = off;
}

static inline dma_addr_t phys_to_dma(struct device *dev, phys_addr_t paddr)
{
	return paddr;
}

static inline phys_addr_t dma_to_phys(struct device *dev, dma_addr_t daddr)
{
	return daddr;
}

static inline void dma_mark_clean(void *addr, size_t size) {}

static inline void set_dma_ops(struct device *dev, const struct dma_map_ops *ops)
{
	dev->archdata.dma_ops = ops;
}

static inline bool dma_capable(struct device *dev, dma_addr_t addr, size_t size)
{
	if (!dev->dma_mask)
		return 0;

	return addr + size - 1 <= *dev->dma_mask;
}

#define HAVE_ARCH_DMA_SET_MASK 1

#include <asm-generic/dma-mapping-common.h>

static inline int
dma_set_mask(struct device *dev, u64 mask)
{
	struct dma_map_ops *dma_ops = get_dma_ops(dev);

	/*
	 * For PCI devices with 64-bit DMA addressing capability, promote
	 * the dma_ops to hybrid, with the consistent memory DMA space limited
	 * to 32-bit. For 32-bit capable devices, limit the streaming DMA
	 * address range to max_direct_dma_addr.
	 */
	if (dma_ops == gx_pci_dma_map_ops ||
	    dma_ops == gx_hybrid_pci_dma_map_ops ||
	    dma_ops == gx_legacy_pci_dma_map_ops) {
		if (mask == DMA_BIT_MASK(64) &&
		    dma_ops == gx_legacy_pci_dma_map_ops)
			set_dma_ops(dev, gx_hybrid_pci_dma_map_ops);
		else if (mask > dev->archdata.max_direct_dma_addr)
			mask = dev->archdata.max_direct_dma_addr;
	}

	if (!dev->dma_mask || !dma_supported(dev, mask))
		return -EIO;

	*dev->dma_mask = mask;

	return 0;
}

/*
 * dma_alloc_noncoherent() is #defined to return coherent memory,
 * so there's no need to do any flushing here.
 */
static inline void dma_cache_sync(struct device *dev, void *vaddr, size_t size,
				  enum dma_data_direction direction)
{
}

#endif /* _ASM_TILE_DMA_MAPPING_H */
